<?php

namespace Tests\Unit;

use App\Models\Order;
use App\Models\OrderDeliverable;
use PHPUnit\Framework\TestCase;

class OrderModelTest extends TestCase
{
    public function test_order_model_has_correct_fillable_fields(): void
    {
        $expectedFillable = [
            'name',
            'idea',
            'grant_area_id',
            'execution_location',
            'start_date',
            'duration_months',
            'target_group_type',
            'state_id',
            'city_id',
            'beneficiaries_count',
            'goals',
            'strengths',
            'risks',
            'declaration_checked',
            'total_amount',
            'org_id',
        ];

        $order = new Order;
        $this->assertEquals($expectedFillable, $order->getFillable());
    }

    public function test_order_model_has_correct_casts(): void
    {
        $order = new Order;
        $casts = $order->getCasts();

        $this->assertEquals('date', $casts['start_date']);
        $this->assertEquals('boolean', $casts['declaration_checked']);
        $this->assertEquals('decimal:2', $casts['total_amount']);
        $this->assertEquals('integer', $casts['duration_months']);
        $this->assertEquals('integer', $casts['beneficiaries_count']);
    }

    public function test_order_validation_rules_exist(): void
    {
        $rules = Order::getValidationRules();

        $this->assertIsArray($rules);
        $this->assertArrayHasKey('name', $rules);
        $this->assertArrayHasKey('declaration_checked', $rules);
        $this->assertEquals('required|accepted', $rules['declaration_checked']);
    }

    public function test_deliverable_model_has_correct_fillable_fields(): void
    {
        $expectedFillable = [
            'order_id',
            'deliverable_name',
            'start_date',
            'end_date',
            'amount',
            'notes',
        ];

        $deliverable = new OrderDeliverable;
        $this->assertEquals($expectedFillable, $deliverable->getFillable());
    }

    public function test_deliverable_validation_rules_exist(): void
    {
        $rules = OrderDeliverable::getValidationRules();

        $this->assertIsArray($rules);
        $this->assertArrayHasKey('deliverable_name', $rules);
        $this->assertArrayHasKey('end_date', $rules);
        $this->assertEquals('required|date|after:start_date', $rules['end_date']);
    }

    public function test_models_have_media_collections(): void
    {
        $order = new Order;

        // Test that Order implements HasMedia
        $this->assertInstanceOf(\Spatie\MediaLibrary\HasMedia::class, $order);
    }
}
