<?php

namespace Tests\Feature;

use App\Models\Association;
use App\Models\Org;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class UserAssociationFormTest extends TestCase
{
    use RefreshDatabase;

    public function test_user_association_assignment_logic(): void
    {
        $org = Org::factory()->create();
        $user = User::factory()->create();
        $association = Association::factory()->create(['org_id' => $org->id]);

        // Associate user with organization
        $user->orgs()->attach($org);

        // Test assignment
        $user->setAssociationInOrg($org, $association);
        $this->assertTrue($user->hasAssociationInOrg($org));
        $this->assertEquals($association->id, $user->getAssociationInOrg($org)->id);
    }

    public function test_user_association_update_logic(): void
    {
        $org = Org::factory()->create();
        $user = User::factory()->create();
        $association1 = Association::factory()->create(['org_id' => $org->id]);
        $association2 = Association::factory()->create(['org_id' => $org->id]);

        // Associate user with organization
        $user->orgs()->attach($org);

        // Set initial association
        $user->setAssociationInOrg($org, $association1);
        $this->assertEquals($association1->id, $user->getAssociationInOrg($org)->id);

        // Update to new association
        $user->setAssociationInOrg($org, $association2);
        $this->assertEquals($association2->id, $user->getAssociationInOrg($org)->id);

        // Verify only one association exists
        $this->assertEquals(1, $user->associations()->wherePivot('org_id', $org->id)->count());
    }

    public function test_user_association_removal_logic(): void
    {
        $org = Org::factory()->create();
        $user = User::factory()->create();
        $association = Association::factory()->create(['org_id' => $org->id]);

        // Associate user with organization and association
        $user->orgs()->attach($org);
        $user->setAssociationInOrg($org, $association);

        $this->assertTrue($user->hasAssociationInOrg($org));

        // Remove association
        $user->setAssociationInOrg($org, null);

        $this->assertFalse($user->hasAssociationInOrg($org));
        $this->assertNull($user->getAssociationInOrg($org));
    }

    public function test_association_form_field_options(): void
    {
        $org1 = Org::factory()->create();
        $org2 = Org::factory()->create();
        $association1 = Association::factory()->create(['org_id' => $org1->id]);
        $association2 = Association::factory()->create(['org_id' => $org2->id]);

        // Test that options are filtered by organization
        $org1Associations = Association::where('org_id', $org1->id)->pluck('name', 'id');
        $org2Associations = Association::where('org_id', $org2->id)->pluck('name', 'id');

        $this->assertCount(1, $org1Associations);
        $this->assertCount(1, $org2Associations);
        $this->assertTrue($org1Associations->has($association1->id));
        $this->assertTrue($org2Associations->has($association2->id));
        $this->assertFalse($org1Associations->has($association2->id));
        $this->assertFalse($org2Associations->has($association1->id));
    }
}
