# 🛠️ Troubleshooting & Frequently Asked Questions

This document collects common issues reported by users of the **Saudi Riyal Font** and provides suggested fixes or work‑arounds.  If your problem is not listed here, please open a new issue on GitHub.

## Symbol alignment appears off (#18)

**Symptom:** The currency symbol appears slightly higher than adjacent text or seems vertically misaligned.

**Cause:** Early releases of the font used generic baseline metrics which could position the glyph slightly above the baseline in some rendering engines.

**Fix:** Version 1.1.0 adjusts the glyph’s baseline to better align with surrounding text.  If you still notice misalignment, wrap the symbol in an inline element and apply a small vertical offset:

```html
<span class="saudi-riyal-wrapper"><span class="icon-saudi_riyal_new"></span></span>
```

```css
.saudi-riyal-wrapper {
  display: inline-block;
  vertical-align: -0.1em; /* adjust this value to suit your design */
}
```

## Currency not showing in `<select>` dropdowns (#16)

**Symptom:** When you place a `<span>` or custom markup inside a `<option>` element, the Saudi Riyal symbol does not appear.

**Cause:** Native `<select>` elements do **not** render HTML inside `<option>` tags; only plain text is permitted.  As a result, CSS classes that rely on pseudo‑elements (`::before`/`::after`) will not work inside `<option>` tags.

**Work‑around:** Use plain text inside the `<option>` and include the Unicode code point directly.  For example:

```html
<select name="currency">
  <option value="SAR">\u20C1 — Saudi Riyal</option>
  <!-- or fallback to \uE900 for older systems -->
  <option value="USD">$ — US Dollar</option>
</select>
```

Alternatively, use a custom select component (for example, a listbox built with `<div>`/`<ul>`) that allows HTML content within options.  You can then apply the `.icon-saudi_riyal*` classes as shown in the README.

## Saudi Riyal symbol not rendering in Magento email templates (#14)

**Symptom:** The currency symbol appears as an HTML entity or doesn’t render at all in Magento transactional email templates.

**Cause:** Most email clients block external CSS and font files for security reasons.  Templates relying on CDN‑hosted fonts or external styles are stripped, causing the symbol to fall back to a missing‑glyph box.

**Solution:** Embed the font directly in the template using Base64 encoding.  Here is an example of how to inline the font in an email:

```html
<style type="text/css">
@font-face {
  font-family: 'saudi_riyal';
  src: url(data:font/truetype;charset=utf-8;base64,AAEAAA…==) format('truetype');
  font-weight: normal;
  font-style: normal;
}
.riyal {
  font-family: 'saudi_riyal';
}
</style>

<!-- Use the new Unicode code point if possible -->
<span class="riyal">⃁</span> 500
```

Replace `AAEAAA…==` with the Base64 encoded contents of `fonts/regular/saudi_riyal.ttf`.  Since many email clients remove `<link>` tags and external resources, embedding the font and the symbol inlined is the most reliable approach.  If embedding fonts is not possible, you can alternatively convert the symbol to an image and embed it inline.

## Bold text not working (#13)

**Symptom:** The symbol does not appear in bold when a `font-weight: bold` style is applied, or browsers simulate the weight rather than using a proper bold glyph.

**Cause:** The original release included only a regular weight, so browsers attempted to synthesize a bold version.  This yielded inconsistent results across platforms.

**Fix:** A true bold font file is now included under `fonts/bold/saudi_riyal.ttf` (and `.woff`, `.woff2`).  Define a separate `@font-face` rule with a distinct `font-family` and `font-weight: bold`.  Then use the provided classes (`.icon-saudi_riyal_bold` or `.icon-saudi_riyal_bold_new`) to display the bold symbol.  See the CSS examples in the README for details.

For frameworks like Odoo that override font families on headings and price classes, explicitly set the font family on the relevant elements:

```css
body,
h1, h2, h3, h4, h5, h6,
.h1, .h2, .h3, .h4, .h5, .h6,
.oe_price {
  font-family: 'saudi_riyal', var(--body-font-family);
}
```

## Symbol missing in PDFs generated by wkhtmltopdf (#12)

**Symptom:** The symbol appears correctly in HTML but disappears or shows a blank box when generating PDFs using wkhtmltopdf (e.g. via `barryvdh/laravel-snappy`).

**Cause:** wkhtmltopdf embeds only the fonts it detects as being used.  If the symbol is referenced via a pseudo‑element or a CSS class and does not appear in visible content during the initial render, the engine may omit the font from the PDF.

**Solutions:**

1. **Use an absolute path** when defining the font in `@font-face`:

   ```css
   @font-face {
     font-family: 'saudi_riyal';
     src: url('file:///absolute/path/to/fonts/regular/saudi_riyal.ttf') format('truetype');
   }
   ```

2. **Embed the font using Base64** as shown in the Magento email example.  This removes the dependency on local file paths.

3. **Preload the font by injecting hidden content.**  wkhtmltopdf will only embed the font if it finds characters rendered during layout.  You can force inclusion by adding a hidden element that uses the symbol:

   ```css
   /* This hidden element forces wkhtmltopdf to include the font */
   .hidden-preload::before {
     content: '\e900'; /* or '\20c1' */
     font-family: 'saudi_riyal';
     position: absolute;
     left: -9999px;
     top: -9999px;
     font-size: 1px;
     width: 1px;
     height: 1px;
     overflow: hidden;
     visibility: hidden;
   }
   ```

   Then include `<div class="hidden-preload"></div>` somewhere in your template.  This invisible glyph ensures the font is embedded in the PDF.

4. **Configure wkhtmltopdf options.**  When using `wkhtmltopdf` directly, pass `--enable-local-file-access` to allow access to local font files.

## Issues with Flutter or React Native (#6 & #15)

**Symptom:** The symbol appears as a question mark (�) in Flutter or React Native, or the digit `1` disappears when combined with the symbol.

**Causes & Fixes:**

* **Font not loaded properly:** Make sure the font is declared correctly in your Flutter `pubspec.yaml` or loaded via Expo’s `useFonts`.  In Flutter, declare both regular and bold weights:

  ```yaml
  flutter:
    fonts:
      - family: SaudiRiyal
        fonts:
          - asset: fonts/regular/saudi_riyal.ttf
          - asset: fonts/bold/saudi_riyal.ttf
            weight: 700
  ```

* **Digits vanish when combined with the symbol:** When a single `Text` widget uses a custom font for the entire string, characters that are not present in the custom font (like digits) may be omitted.  Use `Text.rich` with `TextSpan` to assign the Saudi Riyal font only to the symbol and use the default system font for the digits:

  ```dart
  Text.rich(
    TextSpan(
      children: [
        TextSpan(
          text: '\u20C1 ',
          style: TextStyle(fontFamily: 'SaudiRiyal', fontSize: 20),
        ),
        TextSpan(
          text: '1,00',
          style: TextStyle(fontFamilyFallback: ['Roboto', 'System'], fontSize: 20),
        ),
      ],
    ),
  );
  ```

* **React Native:** See the React Native/Expo notes in the README.  Install a URL polyfill or load the `.ttf` files directly and set the font family only on the symbol.  Use a fallback font for the rest of the text to avoid missing characters.

## Placeholder character and Unicode upgrade (#9)

**Context:** The original font used the private‑use code point `U+E900` because no official code point existed for the Saudi Riyal sign.  The Unicode Technical Committee has recommended encoding the sign at `U+20C1` in the currency symbols block.  Unicode 17 is scheduled for release in September 2025.

**Recommendation:** Use the new `U+20C1` code point (`⃁`) for all new projects.  The font will map that code point to the correct glyph.  For backward compatibility with systems that do not yet support Unicode 17, you can continue to reference `U+E900`.  The package provides separate CSS classes for each code point and weight:

* `.icon-saudi_riyal`, `.icon-saudi_riyal_bold` – use U+E900 (legacy)
* `.icon-saudi_riyal_new`, `.icon-saudi_riyal_bold_new` – use U+20C1 (new standard)

Update your applications accordingly to ensure a smooth transition once Unicode 17 is widely available.

## Need more help?

If you encounter an issue that is not addressed here, please search the open [issues](https://github.com/emran-alhaddad/Saudi-Riyal-Font/issues) or open a new one.  Providing a minimal reproducible example will help maintainers diagnose and fix problems more effectively.

