<?php

namespace Database\Seeders;

use App\Models\Org;
use App\Models\ProjectCharter;
use App\Models\ProjectCharterDeliverable;
use App\Models\ProjectCharterDeliverableTask;
use App\Models\User;
use Illuminate\Database\Seeder;

class ProjectCharterDeliverableSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $org = Org::first() ?? Org::create(['name' => 'مؤسسة سدو']);
        $user = User::first();
        $charters = ProjectCharter::all();

        if ($charters->isEmpty()) {
            $this->command->warn('No project charters found. Please run ProjectCharterSeeder first.');

            return;
        }

        // Deliverable templates for different types of projects
        $deliverableTemplates = [
            'تطوير' => [
                ['name' => 'دراسة الجدوى والتحليل الأولي', 'weight' => 15, 'cost_range' => [20000, 40000]],
                ['name' => 'التصميم والتطوير', 'weight' => 40, 'cost_range' => [60000, 120000]],
                ['name' => 'الاختبار والتجريب', 'weight' => 25, 'cost_range' => [30000, 60000]],
                ['name' => 'التوثيق والتدريب', 'weight' => 20, 'cost_range' => [15000, 30000]],
            ],
            'تدريب' => [
                ['name' => 'إعداد المناهج والمواد التدريبية', 'weight' => 30, 'cost_range' => [25000, 50000]],
                ['name' => 'تنفيذ البرامج التدريبية', 'weight' => 50, 'cost_range' => [40000, 80000]],
                ['name' => 'التقييم والمتابعة', 'weight' => 20, 'cost_range' => [10000, 25000]],
            ],
            'دعم' => [
                ['name' => 'تقييم الاحتياجات والأولويات', 'weight' => 20, 'cost_range' => [15000, 30000]],
                ['name' => 'توزيع المساعدات والخدمات', 'weight' => 60, 'cost_range' => [50000, 150000]],
                ['name' => 'المتابعة وقياس الأثر', 'weight' => 20, 'cost_range' => [10000, 25000]],
            ],
            'استثمار' => [
                ['name' => 'دراسة الفرص الاستثمارية', 'weight' => 25, 'cost_range' => [30000, 60000]],
                ['name' => 'تنفيذ الاستثمارات', 'weight' => 55, 'cost_range' => [100000, 300000]],
                ['name' => 'مراقبة الأداء والعوائد', 'weight' => 20, 'cost_range' => [20000, 40000]],
            ],
        ];

        // Task templates for deliverables
        $taskTemplates = [
            'دراسة' => [
                'جمع وتحليل البيانات',
                'إعداد التقرير الأولي',
                'مراجعة النتائج مع الفريق',
                'إعداد التقرير النهائي',
            ],
            'تطوير' => [
                'تحديد المتطلبات الفنية',
                'إعداد التصميم التفصيلي',
                'تطوير النموذج الأولي',
                'اختبار الأداء والجودة',
            ],
            'تدريب' => [
                'إعداد المحتوى التدريبي',
                'تنسيق مع المدربين',
                'تنفيذ الجلسات التدريبية',
                'تقييم نتائج التدريب',
            ],
            'توزيع' => [
                'تحديد المستفيدين المؤهلين',
                'تجهيز وتعبئة المساعدات',
                'تنفيذ عملية التوزيع',
                'متابعة وتقييم الأثر',
            ],
        ];

        $createdDeliverables = [];
        $createdTasks = [];

        foreach ($charters as $charter) {
            // Determine deliverable type based on charter name
            $charterType = 'تطوير'; // default
            if (str_contains($charter->name, 'تدريب') || str_contains($charter->name, 'أكاديمية')) {
                $charterType = 'تدريب';
            } elseif (str_contains($charter->name, 'دعم') || str_contains($charter->name, 'مساعدات') || str_contains($charter->name, 'سلال')) {
                $charterType = 'دعم';
            } elseif (str_contains($charter->name, 'استثمار') || str_contains($charter->name, 'صندوق')) {
                $charterType = 'استثمار';
            }

            $deliverableData = $deliverableTemplates[$charterType] ?? $deliverableTemplates['تطوير'];

            foreach ($deliverableData as $template) {
                // Calculate delivery date within charter timeframe
                $charterStart = $charter->start_date;
                $charterEnd = $charter->expected_end_date;
                $deliveryDate = fake()->dateTimeBetween($charterStart, $charterEnd);

                $deliverable = ProjectCharterDeliverable::create([
                    'project_charter_id' => $charter->id,
                    'name' => $template['name'],
                    'description' => 'وصف تفصيلي لمخرجات '.$template['name'].' ضمن مشروع '.$charter->name,
                    'delivery_date' => $deliveryDate,
                    'weight' => $template['weight'],
                    'cost' => fake()->randomFloat(2, $template['cost_range'][0], $template['cost_range'][1]),
                    'created_by' => $user->id,
                    'org_id' => $org->id,
                ]);

                $createdDeliverables[] = $deliverable;

                // Create 2-4 tasks per deliverable
                $numTasks = fake()->numberBetween(2, 4);
                $taskType = 'دراسة'; // default

                if (str_contains($template['name'], 'تطوير') || str_contains($template['name'], 'تصميم')) {
                    $taskType = 'تطوير';
                } elseif (str_contains($template['name'], 'تدريب')) {
                    $taskType = 'تدريب';
                } elseif (str_contains($template['name'], 'توزيع')) {
                    $taskType = 'توزيع';
                }

                $availableTasks = $taskTemplates[$taskType] ?? $taskTemplates['دراسة'];
                $selectedTasks = fake()->randomElements($availableTasks, $numTasks);

                foreach ($selectedTasks as $index => $taskName) {
                    // Calculate task dates within deliverable timeframe
                    $taskStartOffset = $index * 7; // Start tasks 1 week apart
                    $taskStart = $charterStart->copy()->addDays($taskStartOffset);
                    $taskDuration = fake()->numberBetween(7, 30); // 1-4 weeks duration
                    $taskEnd = $taskStart->copy()->addDays($taskDuration);

                    // Ensure task end doesn't exceed deliverable delivery date
                    if ($taskEnd > $deliveryDate) {
                        $taskEnd = $deliveryDate;
                    }

                    $task = ProjectCharterDeliverableTask::create([
                        'project_charter_deliverable_id' => $deliverable->id,
                        'name' => $taskName,
                        'description' => 'تفاصيل تنفيذ مهمة '.$taskName.' ضمن مخرج '.$deliverable->name,
                        'start_date' => $taskStart,
                        'end_date' => $taskEnd,
                        'progress_percentage' => fake()->randomFloat(2, 0, 100),
                        'cost' => fake()->randomFloat(2, 2000, 15000),
                        'created_by' => $user->id,
                        'org_id' => $org->id,
                    ]);

                    $createdTasks[] = $task;
                }
            }
        }

        $this->command->info('تم إنشاء '.count($createdDeliverables).' مخرج و '.count($createdTasks).' مهمة لـ '.count($charters).' ميثاق مشروع');

        // Summary by charter type
        $summary = collect($createdDeliverables)->groupBy(function ($deliverable) {
            $charter = $deliverable->projectCharter;
            if (str_contains($charter->name, 'تدريب') || str_contains($charter->name, 'أكاديمية')) {
                return 'تدريب';
            } elseif (str_contains($charter->name, 'دعم') || str_contains($charter->name, 'مساعدات')) {
                return 'دعم';
            } elseif (str_contains($charter->name, 'استثمار') || str_contains($charter->name, 'صندوق')) {
                return 'استثمار';
            }

            return 'تطوير';
        });

        foreach ($summary as $type => $deliverables) {
            $this->command->info("  - {$type}: {$deliverables->count()} deliverables");
        }
    }
}
