<?php

namespace Database\Seeders;

use App\Models\GrantArea;
use App\Models\Initiative;
use App\Models\PerformanceCard;
use App\Models\User;
use Illuminate\Database\Seeder;

class InitiativeSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $user = User::first();
        $performanceCards = PerformanceCard::all();

        if ($performanceCards->isEmpty()) {
            $this->command->warn('No performance cards found. Please run PerformanceCardSeeder first.');

            return;
        }

        $initiativesData = [
            // Initiatives for developmental card (تطوير قدرات الموظفين)
            [
                'name' => 'برنامج التدريب التقني المتقدم',
                'description' => 'تنفيذ برامج تدريبية متخصصة في التقنيات الحديثة والذكاء الاصطناعي لتطوير مهارات الموظفين',
                'card_goal_keyword' => 'تطوير قدرات الموظفين',
            ],
            [
                'name' => 'مختبر الابتكار المؤسسي',
                'description' => 'إنشاء مساحة عمل تفاعلية لتشجيع الابتكار وتطوير الحلول الإبداعية داخل المؤسسة',
                'card_goal_keyword' => 'تطوير قدرات الموظفين',
            ],

            // Initiatives for support card (دعم الأسر المحتاجة)
            [
                'name' => 'برنامج المساعدات المالية الطارئة',
                'description' => 'تقديم دعم مالي سريع للأسر التي تواجه ظروف طارئة أو أزمات اقتصادية',
                'card_goal_keyword' => 'دعم الأسر المحتاجة',
            ],
            [
                'name' => 'مشروع السلال الغذائية',
                'description' => 'توزيع سلال غذائية شهرية تحتوي على المواد الأساسية للأسر المحتاجة',
                'card_goal_keyword' => 'دعم الأسر المحتاجة',
            ],

            // Initiatives for empowerment card (تمكين الشباب)
            [
                'name' => 'أكاديمية القيادة الشبابية',
                'description' => 'برنامج تدريبي شامل لتطوير المهارات القيادية والإدارية للشباب',
                'card_goal_keyword' => 'تمكين الشباب',
            ],
            [
                'name' => 'حاضنة المشاريع الشبابية',
                'description' => 'دعم الشباب في تأسيس وتطوير مشاريعهم الريادية من خلال التوجيه والتمويل',
                'card_goal_keyword' => 'تمكين الشباب',
            ],

            // Initiatives for investment card (استثمار الموارد)
            [
                'name' => 'صندوق الاستثمار المستدام',
                'description' => 'إنشاء محفظة استثمارية متنوعة تركز على المشاريع المستدامة والمربحة طويلة المدى',
                'card_goal_keyword' => 'استثمار الموارد المالية',
            ],
            [
                'name' => 'مشروع الطاقة المتجددة',
                'description' => 'الاستثمار في تقنيات الطاقة الشمسية وطاقة الرياح لتحقيق عوائد مستدامة',
                'card_goal_keyword' => 'استثمار الموارد المالية',
            ],
        ];

        $createdInitiatives = [];

        foreach ($initiativesData as $initiativeData) {
            // Find the appropriate performance card based on the keyword
            $performanceCard = $performanceCards->first(function ($card) use ($initiativeData) {
                return str_contains($card->goal, $initiativeData['card_goal_keyword']);
            });

            if (! $performanceCard) {
                continue;
            }

            // Use the performance card's wallet dates if available, otherwise use defaults
            $wallet = $performanceCard->wallet;
            $baseStartDate = $wallet ? $wallet->start_date : now();
            $baseEndDate = $wallet ? $wallet->expected_end_date : now()->addMonths(12);

            $startDate = fake()->dateTimeBetween($baseStartDate, $baseStartDate->copy()->addMonths(2));
            $endDate = fake()->dateTimeBetween($startDate, $baseEndDate);

            $initiative = Initiative::create([
                'performance_card_id' => $performanceCard->id,
                'name' => $initiativeData['name'],
                'description' => $initiativeData['description'],
                'start_date' => $startDate,
                'expected_end_date' => $endDate,
                'created_by' => $user->id,
            ]);

            // Attach random grant areas (1-2 per initiative)
            $grantAreas = GrantArea::inRandomOrder()->limit(rand(1, 2))->get();
            if ($grantAreas->count() > 0) {
                $initiative->grantAreas()->attach($grantAreas->pluck('id'));
            }

            $createdInitiatives[] = $initiative;
        }

        $this->command->info('تم إنشاء '.count($createdInitiatives).' مبادرة عبر '.$performanceCards->count().' بطاقة أداء');
    }
}
