<?php

namespace Database\Factories;

use App\Enums\RiskLevel;
use App\Models\ProjectCharter;
use App\Models\ProjectCharterRisk;
use Illuminate\Database\Eloquent\Factories\Factory;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<\App\Models\ProjectCharterRisk>
 */
class ProjectCharterRiskFactory extends Factory
{
    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        $incidentLikelihood = fake()->randomElement(RiskLevel::cases());
        $impactLevel = fake()->randomElement(RiskLevel::cases());

        return [
            'name' => fake()->randomElement([
                'تأخير المورد الرئيسي',
                'نقص في الموارد البشرية المتخصصة',
                'تغيير في المتطلبات الحكومية',
                'مشاكل تقنية في الأنظمة',
                'تأخير في الموافقات الإدارية',
                'تجاوز الميزانية المحددة',
                'عدم توفر المواد الخام',
                'مشاكل في سلسلة التوريد',
                'تغيير في الأولويات المؤسسية',
                'مقاومة التغيير من الموظفين',
                'مشاكل في البنية التحتية',
                'عدم استقرار الوضع الاقتصادي',
            ]),
            'description' => fake()->randomElement([
                'قد يحدث تأخير في تسليم المواد أو الخدمات من الموردين الخارجيين',
                'نقص في عدد الموظفين المؤهلين للعمل على المشروع',
                'احتمالية تغيير اللوائح أو المتطلبات الحكومية أثناء تنفيذ المشروع',
                'مواجهة صعوبات تقنية أو فنية في تطبيق الأنظمة المطلوبة',
                'تأخير في الحصول على الموافقات اللازمة من الجهات المختصة',
                'زيادة التكاليف عن الميزانية المحددة للمشروع',
                'عدم توفر المواد أو المعدات المطلوبة في الوقت المناسب',
                'مشاكل في عمليات الشراء والتوريد',
                'تغيير أولويات المؤسسة مما يؤثر على المشروع',
                'مقاومة من الموظفين للتغييرات المطلوبة',
                'مشاكل في الشبكات أو المرافق التحتية',
                'تذبذب في الأوضاع الاقتصادية مما يؤثر على المشروع',
            ]),
            'incident_likelihood' => $incidentLikelihood,
            'impact_level' => $impactLevel,
            'impact_on_cost' => fake()->randomFloat(2, 5000, 100000),
            'impact_on_time' => fake()->numberBetween(5, 90), // days
            'mitigation_plan' => fake()->randomElement([
                'إعداد قائمة بالموردين البديلين والتعاقد المسبق معهم',
                'تدريب موظفين إضافيين وتوظيف خبراء خارجيين عند الحاجة',
                'المتابعة المستمرة مع الجهات الحكومية والحصول على تأكيدات مسبقة',
                'إجراء اختبارات شاملة للأنظمة ووضع خطط بديلة',
                'البدء المبكر في إجراءات الموافقات والمتابعة المستمرة',
                'وضع احتياطي مالي إضافي ومراقبة التكاليف بانتظام',
                'تأمين مصادر بديلة للمواد والتخزين المسبق للمواد الحرجة',
                'تنويع الموردين ووضع عقود إطارية مع موردين متعددين',
                'التواصل المستمر مع الإدارة العليا وتوضيح أهمية المشروع',
                'برامج توعية وتدريب للموظفين وإشراكهم في عملية التخطيط',
                'إجراء صيانة وقائية منتظمة ووضع خطط طوارئ للبنية التحتية',
                'مراقبة المؤشرات الاقتصادية ووضع خطط مرونة في التنفيذ',
            ]),
            'contingency_plan' => fake()->randomElement([
                'تفعيل العقود مع الموردين البديلين فوراً',
                'الاستعانة بشركات خارجية متخصصة',
                'التواصل مع الجهات المختصة لتسريع الإجراءات',
                'تطبيق الحلول التقنية البديلة المعدة مسبقاً',
                'تسريع عمليات المتابعة وتقديم طلبات عاجلة',
                'تطبيق خطة التوفير وإعادة توزيع الموارد',
                'تفعيل المصادر البديلة واستخدام المخزون الاحتياطي',
                'التبديل إلى الموردين البديلين',
                'إعادة تقييم أولويات المشروع مع الإدارة العليا',
                'تطبيق خطة إدارة التغيير المكثفة',
                'تفعيل خطة الطوارئ للبنية التحتية',
                'تطبيق خطة المرونة وتأجيل الأنشطة غير الحرجة',
            ]),
            'financial_contingency' => fake()->randomFloat(2, 2000, 50000),
            'technical_contingency' => fake()->randomFloat(2, 1000, 25000),
        ];
    }

    /**
     * Configure the model factory.
     */
    public function configure(): static
    {
        return $this->afterMaking(function (ProjectCharterRisk $risk) {
            if (! $risk->org_id && $risk->project_charter_id) {
                $charter = ProjectCharter::find($risk->project_charter_id);
                if ($charter) {
                    $risk->org_id = $charter->org_id;

                    // Set created_by to a user from the same org or charter creator
                    if (! $risk->created_by) {
                        $org = $charter->org;
                        $orgUsers = $org ? $org->users()->get() : collect();

                        if ($orgUsers->isNotEmpty()) {
                            $risk->created_by = $orgUsers->random()->id;
                        } else {
                            $risk->created_by = $charter->created_by;
                        }
                    }
                }
            }
        });
    }

    /**
     * High priority risk state.
     */
    public function highRisk(): static
    {
        return $this->state(fn (array $attributes) => [
            'incident_likelihood' => RiskLevel::HIGH,
            'impact_level' => RiskLevel::HIGH,
            'impact_on_cost' => fake()->randomFloat(2, 50000, 200000),
            'impact_on_time' => fake()->numberBetween(30, 120),
        ]);
    }

    /**
     * Low priority risk state.
     */
    public function lowRisk(): static
    {
        return $this->state(fn (array $attributes) => [
            'incident_likelihood' => RiskLevel::LOW,
            'impact_level' => RiskLevel::LOW,
            'impact_on_cost' => fake()->randomFloat(2, 1000, 10000),
            'impact_on_time' => fake()->numberBetween(1, 15),
        ]);
    }
}
