# Workflow System Documentation

## Overview

The workflow system has been refactored to be clean, generalized, and reusable across any model in the application. The system consists of two main components:

## Components

### 1. HasWorkflow Trait (`app/Workflows/Traits/HasWorkflow.php`)

This trait provides all the necessary workflow functionality to any model:

**Methods provided:**
- `workflowInstance()` - Polymorphic relationship to WorkflowInstance
- `currentStep()` - Get the current workflow step
- `hasWorkflow()` - Check if model has an active workflow
- `hasPendingStep()` - Check if workflow has a pending step
- `getWorkflowStatusLabel()` - Get human-readable workflow status

### 2. WorkflowActions Class (`app/Workflows/Actions/WorkflowActions.php`)

This class provides pre-built Filament actions for workflow operations:

**Static methods:**
- `getActions($record, $refreshCallback)` - Get all workflow actions
- `getStartWorkflowAction($record, $refreshCallback)` - Get start workflow action
- `getReviewStepAction($record, $refreshCallback)` - Get review/approve step action
- `getRejectStepAction($record, $refreshCallback)` - Get reject step action
- `autoInitializeWorkflow($record)` - Auto-initialize workflow in mount() methods

## Usage

### Adding Workflow to a Model

1. **Add the trait to your model:**
```php
<?php

namespace App\Models;

use App\Workflows\Traits\HasWorkflow;
use Illuminate\Database\Eloquent\Model;

class YourModel extends Model
{
    use HasWorkflow;
    
    // Your model code...
}
```

2. **Add your model to the workflows config:**
```php
// config/workflows.php
'models' => [
    \App\Models\YourModel::class => 'YourModel',
],

'required_steps' => [
    \App\Models\YourModel::class => [
        'review',
        'approval',
        'completion',
    ],
],
```

### Adding Workflow to Filament Pages

1. **In your ViewRecord page:**
```php
<?php

namespace App\Filament\Admin\Resources\YourModelResource\Pages;

use App\Workflows\Actions\WorkflowActions;
use Filament\Actions;
use Filament\Resources\Pages\ViewRecord;

class ViewYourModel extends ViewRecord
{
    public function mount(int|string $record): void
    {
        parent::mount($record);
        
        // Auto-initialize workflow if none exists
        WorkflowActions::autoInitializeWorkflow($this->record);
    }

    protected function getHeaderActions(): array
    {
        return [
            Actions\EditAction::make(),
            // Add all workflow actions with one line
            ...WorkflowActions::getActions($this->record, fn($keys) => $this->refreshFormData($keys)),
        ];
    }
}
```

## Benefits

1. **Clean Code:** Models only need the trait, no duplicate workflow methods
2. **Reusability:** Any model can have workflows with minimal setup
3. **Consistency:** All workflow actions behave the same across models
4. **Maintainability:** Workflow logic is centralized in the Actions class
5. **Flexibility:** Each model can have different workflow steps in config

## Migration from Old System

The Order model has been updated to use this new system:

**Before:**
- Had custom `workflowInstance()` and `currentStep()` methods
- ViewOrder had custom workflow actions (100+ lines)

**After:**
- Uses `HasWorkflow` trait (1 line)
- ViewOrder uses `WorkflowActions::getActions()` (1 line)
- Reduced ViewOrder from 200+ lines to ~30 lines

## Future Models

Any future model that needs workflow support can be added with:
1. Add `HasWorkflow` trait to the model
2. Add model to `config/workflows.php`
3. Use `WorkflowActions::getActions()` in Filament pages

This ensures consistency and reduces code duplication across the application.