<?php

namespace App\Filament\Admin\Resources\Users\Forms;

use App\Models\Association;
use App\Models\Role;
use Filament\Facades\Filament;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;

class UserForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make(__('common.user_information'))
                    ->description(__('common.user_information_description'))
                    ->schema([
                        TextInput::make('name')
                            ->label(__('common.name'))
                            ->placeholder(__('common.name_placeholder'))
                            ->maxLength(255)
                            ->required(),

                        TextInput::make('email')
                            ->label(__('common.email'))
                            ->placeholder(__('common.email_placeholder'))
                            ->email()
                            ->required()
                            ->maxLength(255)
                            ->unique(),

                        TextInput::make('phone')
                            ->label(__('common.phone'))
                            ->placeholder(__('common.phone_placeholder'))
                            ->tel()
                            ->maxLength(20),

                        TextInput::make('password')
                            ->label(__('common.password'))
                            ->placeholder(__('common.password_placeholder'))
                            ->password()
                            ->dehydrated(fn ($state) => filled($state))
                            ->required(fn (string $context): bool => $context === 'create')
                            ->minLength(8)
                            ->maxLength(255),
                    ])
                    ->columns(2),

                Section::make(__('common.roles_permissions'))
                    ->description(__('common.roles_permissions_description'))
                    ->schema([
                        Select::make('roles')
                            ->label(__('common.roles'))
                            ->options(function () {
                                $tenant = Filament::getTenant();
                                if (! $tenant) {
                                    return [];
                                }

                                return Role::where('org_id', $tenant->id)
                                    ->pluck('name', 'id');
                            })
                            ->multiple()
                            ->preload()
                            ->searchable()
                            ->afterStateHydrated(function (Select $component, $state, $record) {
                                if ($record) {
                                    $tenant = Filament::getTenant();
                                    if ($tenant) {
                                        // Get roles for this user in the current organization
                                        $roleIds = $record->roles()->pluck('id')->toArray();
                                        $component->state($roleIds);
                                    }
                                }
                            }),

                        Select::make('orgs')
                            ->label(__('common.organizations'))
                            ->relationship('orgs', 'name')
                            ->multiple()
                            ->preload()
                            ->searchable(),

                        Select::make('association_id')
                            ->label(__('common.association'))
                            ->placeholder(__('common.select_association'))
                            ->options(function () {
                                $tenant = Filament::getTenant();
                                if (! $tenant) {
                                    return [];
                                }

                                return Association::where('org_id', $tenant->id)
                                    ->pluck('name', 'id');
                            })
                            ->searchable()
                            ->helperText(__('common.association_helper_text'))
                            ->afterStateHydrated(function (Select $component, $state, $record) {
                                if ($record) {
                                    $tenant = Filament::getTenant();
                                    if ($tenant) {
                                        $association = $record->getAssociationInOrg($tenant);
                                        $component->state($association?->id);
                                    }
                                }
                            }),
                    ])
                    ->columns(2),
            ]);
    }
}
