<?php

namespace App\Filament\Admin\Resources\ProjectCharters\Schemas;

use Filament\Infolists\Components\TextEntry;
use Filament\Schemas\Components\Flex;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Filament\Support\Enums\FontWeight;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\HtmlString;
use RalphJSmit\Filament\Activitylog\Filament\Infolists\Components\Timeline;
use Spatie\Activitylog\Models\Activity;

class ProjectCharterInfolist
{
    public static function configure(Schema $schema): Schema
    {
        return $schema
            ->components([
                Flex::make([
                    // Left Column - Charter Overview
                    Grid::make(1)
                        ->schema([
                            // Charter Overview Card
                            Section::make(__('project_charter.charter_overview'))
                                ->icon('heroicon-o-document-text')
                                ->schema([
                                    Grid::make(2)
                                        ->schema([
                                            TextEntry::make('id')
                                                ->label(__('project_charter.charter_id'))
                                                ->weight(FontWeight::SemiBold)
                                                ->color('gray')
                                                ->badge()
                                                ->prefix('#'),

                                            TextEntry::make('workflow_status')
                                                ->label(__('project_charter.status'))
                                                ->getStateUsing(function ($record) {
                                                    return $record->getWorkflowStatusLabel();
                                                })
                                                ->badge(),

                                            TextEntry::make('name')
                                                ->label(__('project_charter.name'))
                                                ->weight(FontWeight::SemiBold)
                                                ->icon('heroicon-o-document-text'),

                                            TextEntry::make('initiative.performanceCard.project_manner')
                                                ->label(__('project_charter.project_manner'))
                                                ->badge()
                                                ->placeholder('-'),
                                        ]),

                                    TextEntry::make('description')
                                        ->label(__('project_charter.description'))
                                        ->html()
                                        ->columnSpanFull()
                                        ->placeholder(__('project_charter.no_description')),

                                ]),

                            // Project Relationships Card
                            Section::make(__('project_charter.project_relationships'))
                                ->icon('heroicon-o-link')
                                ->schema([
                                    Grid::make(2)
                                        ->schema([
                                            TextEntry::make('initiativeProject.name')
                                                ->label(__('project_charter.initiative_project'))
                                                ->weight(FontWeight::Medium)
                                                ->icon('heroicon-o-rocket-launch')
                                                ->color('primary'),

                                            // TextEntry::make('order.id')
                                            //     ->label(__('project_charter.linked_order'))
                                            //     ->formatStateUsing(fn($record) => $record->order ?
                                            //         "#{$record->order->id} - " . Order::getOrderTypes()[$record->order->type] : '-')
                                            //     ->url(fn($record) => $record->order ? OrderResource::getUrl('view', ['record' => $record->order]) : null)
                                            //     ->openUrlInNewTab(false)
                                            //     ->icon('heroicon-o-document-duplicate')
                                            //     ->color('info')
                                            //     ->placeholder('-'),

                                            TextEntry::make('initiative.name')
                                                ->label(__('project_charter.initiative'))
                                                ->icon('heroicon-o-light-bulb')
                                                ->color('warning'),

                                            TextEntry::make('initiative.performanceCard.department.name')
                                                ->label(__('project_charter.department'))
                                                ->icon('heroicon-o-building-office')
                                                ->color('success')
                                                ->placeholder('-'),

                                            // service provider if exists
                                            // TextEntry::make('serviceProvider.name')
                                            //     ->label(__('service_provider.service_provider'))
                                            //     ->icon('heroicon-o-user')
                                            //     ->color('primary')
                                            //     ->placeholder('-'),
                                            //     // ->url(fn($record) => $record->serviceProvider ? ServiceProviderResource::getUrl('view', ['record' => $record->serviceProvider]) : null),
                                        ]),
                                ]),
                        ]),

                    // Right Column - Timeline & Financial Information
                    Grid::make(1)
                        ->schema([
                            // Timeline Card
                            Section::make(__('project_charter.timeline'))
                                ->icon('heroicon-o-calendar-days')
                                ->schema([
                                    Grid::make(2)
                                        ->schema([
                                            TextEntry::make('start_date')
                                                ->label(__('project_charter.start_date'))
                                                ->date('d M Y')
                                                ->icon('heroicon-o-calendar-days')
                                                ->color('success')
                                                ->weight(FontWeight::Medium),

                                            TextEntry::make('expected_end_date')
                                                ->label(__('project_charter.expected_end_date'))
                                                ->date('d M Y')
                                                ->icon('heroicon-o-calendar')
                                                ->color('warning')
                                                ->weight(FontWeight::Medium),
                                        ]),

                                    // Duration calculation
                                    TextEntry::make('duration')
                                        ->label(__('project_charter.duration'))
                                        ->state(function ($record) {
                                            if (! $record->start_date || ! $record->expected_end_date) {
                                                return '-';
                                            }
                                            $duration = $record->start_date->diffInDays($record->expected_end_date);

                                            return __('project_charter.duration_days', ['days' => $duration]);
                                        })
                                        ->icon('heroicon-o-clock')
                                        ->color('info')
                                        ->columnSpanFull(),
                                ]),

                            // Financial Information Card
                            Section::make(__('project_charter.financial_information'))
                                ->icon('heroicon-o-banknotes')
                                ->schema([
                                    Grid::make(2)
                                        ->schema([
                                            TextEntry::make('total_cost')
                                                ->label(__('project_charter.total_cost'))
                                                ->numeric(2)
                                                ->prefix(new HtmlString('<span class="icon-saudi_riyal"></span>'))
                                                ->icon('heroicon-o-banknotes')
                                                ->color('success')
                                                ->weight(FontWeight::Bold),
                                        ]),
                                ]),

                        ]),
                ])
                    ->columnSpanFull(),

                // Activity Timeline - Full Width Section
                Section::make(__('workflow.general.actions'))
                    ->icon('heroicon-o-clock')
                    ->schema([
                        Timeline::make('activities')
                            ->modifyActivitiesQueryUsing(function (Builder $query) {
                                return $query->whereIn('event', ['workflow_step_approved', 'workflow_step_rejected', 'workflow_completed']);
                            })
                            ->itemIcons([
                                'workflow_step_approved' => 'heroicon-o-check-circle',
                                'workflow_step_rejected' => 'heroicon-o-x-circle',
                                'workflow_completed' => 'heroicon-o-flag',
                            ])
                            ->itemIconColors([
                                'workflow_step_approved' => 'info',
                                'workflow_step_rejected' => 'danger',
                                'workflow_completed' => 'success',
                            ])
                            ->eventDescriptions([
                                'workflow_step_approved' => function (Activity $activity) {
                                    $from_step = $activity->getExtraProperty('from_step');
                                    $to_step = $activity->getExtraProperty('to_step');

                                    $description = __('workflow.activity.step_approved_by', [
                                        'name' => $activity->causer?->name,
                                        'from_step' => $from_step,
                                    ]);
                                    if ($from_step && $to_step) {
                                        $description .= '<br/>'.__('workflow.activity.step_status_changed', ['from_step' => $from_step, 'to_step' => $to_step]);
                                    }

                                    return new HtmlString($description);
                                },
                                'workflow_step_rejected' => function (Activity $activity) {
                                    $step = $activity->getExtraProperty('step');
                                    $description = __('workflow.activity.step_rejected_by', [
                                        'name' => $activity->causer?->name,
                                        'from_step' => $step,
                                    ]);
                                    if ($step) {
                                        $description .= ' '.__('workflow.activity.step_status_was', ['status' => $step]);
                                    }

                                    return new HtmlString($description);
                                },
                                'workflow_completed' => function (Activity $activity) {
                                    $description = __('workflow.activity.workflow_completed_by', ['name' => $activity->causer?->name]);

                                    return new HtmlString($description);
                                },
                            ])
                            ->itemBadges([
                                'workflow_step_approved' => function (Activity $activity) {
                                    $notes = $activity->getExtraProperty('notes');

                                    return $notes ? __('workflow.activity.workflow_notes', ['notes' => $notes]) : __('workflow.activity.workflow_notes_na');
                                },
                                'workflow_step_rejected' => function (Activity $activity) {
                                    $notes = $activity->getExtraProperty('notes');

                                    return $notes ? __('workflow.activity.workflow_notes', ['notes' => $notes]) : __('workflow.activity.workflow_notes_na');
                                },
                                'workflow_completed' => function (Activity $activity) {
                                    $notes = $activity->getExtraProperty('notes');

                                    return $notes ? __('workflow.activity.workflow_notes', ['notes' => $notes]) : __('workflow.activity.workflow_notes_na');
                                },
                            ])
                            ->itemBadgeColors([
                                'workflow_step_approved' => 'info',
                                'workflow_step_rejected' => 'danger',
                                'workflow_completed' => 'success',
                            ]),
                    ])
                    ->collapsible()
                    ->columnSpanFull(),
            ]);
    }
}
