<?php

namespace App\Filament\Admin\Resources\PerformanceCards\Resources\Initiatives\RelationManagers;

use App\Filament\Admin\Resources\PerformanceCards\Resources\Initiatives\Resources\ProjectCharters\ProjectCharterResource;
use Filament\Actions\Action;
use Filament\Actions\AssociateAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\CreateAction;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\DissociateAction;
use Filament\Actions\DissociateBulkAction;
use Filament\Actions\EditAction;
use Filament\Actions\ForceDeleteAction;
use Filament\Actions\ForceDeleteBulkAction;
use Filament\Actions\RestoreAction;
use Filament\Actions\RestoreBulkAction;
use Filament\Facades\Filament;
use Filament\Forms\Components\TextInput;
use Filament\Notifications\Notification;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Components\Utilities\Set;
use Filament\Schemas\Schema;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\Summarizers\Sum;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\HtmlString;

class ProjectsRelationManager extends RelationManager
{
    protected static string $relationship = 'projects';

    public static function getTitle(Model $ownerRecord, string $pageClass): string
    {
        return __('initiative_project.projects');
    }

    public static function getModelLabel(): string
    {
        return __('initiative_project.project');
    }

    public static function getPluralModelLabel(): string
    {
        return __('initiative_project.projects');
    }

    public function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                TextInput::make('name')
                    ->label(__('initiative_project.name'))
                    ->required()
                    ->maxLength(255),
                TextInput::make('weight')
                    ->label(__('initiative_project.weight'))
                    ->required()
                    ->numeric()
                    ->minValue(0.01)
                    ->maxValue(100)
                    ->suffix('%')
                    ->helperText(__('initiative_project.weight_helper'))
                    ->live(onBlur: true)
                    ->afterStateUpdated(function ($state, Get $get, Set $set, ?string $operation, ?object $record) {
                        if ($state && is_numeric($state)) {

                            // Get current total weight (excluding current record if editing)
                            $currentTotal = $this->getOwnerRecord()
                                ->projects()
                                ->when($record, fn ($query) => $query->where('id', '!=', $record->id))
                                ->sum('weight');

                            $newTotal = $currentTotal + (float) $state;
                            $available = 100 - $currentTotal;

                            if ($newTotal > 100) {
                                $set('weight', $available > 0 ? $available : 0.01);

                                Notification::make()
                                    ->title(__('initiative_project.weight_exceeded_title'))
                                    ->body(__('initiative_project.weight_exceeded_message', [
                                        'requested' => $state,
                                        'available' => $available,
                                        'current' => $currentTotal,
                                    ]))
                                    ->warning()
                                    ->send();
                            }
                        }
                    })
                    ->columns(2),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('name')
            ->columns([
                TextColumn::make('name')
                    ->label(__('initiative_project.name'))
                    ->sortable()
                    ->searchable(),
                TextColumn::make('weight')
                    ->label(__('initiative_project.weight'))
                    ->numeric()
                    ->sortable()
                    ->suffix('%')
                    ->summarize(
                        Sum::make()
                            ->label(__('initiative_project.total_weight'))
                            ->formatStateUsing(function ($state) {
                                $color = $state > 100 ? 'danger' : ($state == 100 ? 'success' : 'warning');

                                return new HtmlString(
                                    '<span class="text-'.$color.'-600 font-bold">'.number_format($state, 1).'%</span>'
                                );
                            })
                    ),
                IconColumn::make('charter')
                    ->label(__('initiative_project.has_charter'))
                    ->boolean()
                    ->getStateUsing(fn ($record) => $record->charter !== null),

                TextColumn::make('created_at')
                    ->label(__('initiative_project.created_at'))
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                TrashedFilter::make(),
            ])
            ->headerActions([
                CreateAction::make(), // TODO: validate weight before creating
                AssociateAction::make(),
            ])
            ->recordActions([
                EditAction::make(), // TODO: validate weight before saving
                // TODO: view charter if exists, or create one if not
                // Action::make('view_charter')
                //     ->label(__('project_charter.view_charter'))
                //     ->icon('heroicon-o-document-text')
                //     ->color('info')
                //     ->url(fn ($record) => $record->projectCharter
                //         ? \App\Filament\Resources\ProjectCharterResource::getUrl('view', ['record' => $record->projectCharter])
                //         : null)
                //     ->visible(fn ($record) => $record->projectCharter !== null),

                Action::make('create_charter')
                    ->label(__('project_charter.create_charter'))
                    ->icon('heroicon-o-plus')
                    ->color('primary')
                    // route to create charter page, which is defined in InitiativeResource as 'project-charters' => ManageProjectCharters::route('/{record}/project-charters'),
                    ->url(fn ($record) => ProjectCharterResource::getUrl('create', [
                        'tenant' => Filament::getTenant()?->id,
                        'performance_card' => $this->getOwnerRecord()->performance_card_id,
                        'initiative' => $this->getOwnerRecord()->id,
                        'initiative_project_id' => $record->id]
                    ))
                    ->visible(fn ($record) => $record->charter === null),

                DissociateAction::make(),
                DeleteAction::make(),
                ForceDeleteAction::make(),
                RestoreAction::make(),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DissociateBulkAction::make(),
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ])
            ->modifyQueryUsing(fn (Builder $query) => $query
                ->withoutGlobalScopes([
                    SoftDeletingScope::class,
                ]));
    }

    public function isReadOnly(): bool
    {
        return false;
    }
}
