<?php

namespace App\Filament\Admin\Resources\Orders\Tables;

use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Actions\ForceDeleteAction;
use Filament\Actions\ForceDeleteBulkAction;
use Filament\Actions\RestoreAction;
use Filament\Actions\RestoreBulkAction;
use Filament\Actions\ViewAction;
use Filament\Forms\Components\DatePicker;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Tables\Table;

class OrderTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('id')
                    ->label(__('order.order_number'))
                    ->badge()
                    ->color('gray')
                    ->prefix('#')
                    ->searchable()
                    ->sortable(),

                TextColumn::make('association.name')
                    ->label(__('order.association'))
                    ->searchable()
                    ->sortable()
                    ->toggleable(),

                TextColumn::make('name')
                    ->label(__('order.project_name'))
                    ->searchable()
                    ->sortable()
                    ->limit(50)
                    ->tooltip(function (TextColumn $column): ?string {
                        $state = $column->getState();

                        if (strlen($state) <= $column->getCharacterLimit()) {
                            return null;
                        }

                        return $state;
                    }),

                TextColumn::make('grantArea.name')
                    ->label(__('order.grant_area'))
                    ->searchable()
                    ->sortable()
                    ->toggleable(),

                TextColumn::make('state.name')
                    ->label(__('order.state'))
                    ->searchable()
                    ->sortable()
                    ->toggleable(),

                TextColumn::make('city.name')
                    ->label(__('order.city'))
                    ->searchable()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('start_date')
                    ->label(__('order.start_date'))
                    ->date()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('duration_months')
                    ->label(__('order.duration'))
                    ->suffix(' '.__('order.months'))
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('beneficiaries_count')
                    ->label(__('order.beneficiaries'))
                    ->numeric()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('total_amount')
                    ->label(__('order.total_amount'))
                    ->money('SAR')
                    ->sortable()
                    ->getStateUsing(fn ($record) => $record->total_amount)
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('deliverables_count')
                    ->label(__('order.deliverables_count'))
                    ->counts('deliverables')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('createdBy.name')
                    ->label(__('common.created_by'))
                    ->searchable()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('created_at')
                    ->label(__('common.created_at'))
                    ->dateTime()
                    ->sortable()
                    ->toggleable(),

                TextColumn::make('updated_at')
                    ->label(__('common.updated_at'))
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('deleted_at')
                    ->label(__('common.deleted_at'))
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                SelectFilter::make('association_id')
                    ->label(__('order.association'))
                    ->relationship('association', 'name')
                    ->searchable()
                    ->preload(),

                SelectFilter::make('grant_area_id')
                    ->label(__('order.grant_area'))
                    ->relationship('grantArea', 'name')
                    ->searchable()
                    ->preload(),

                SelectFilter::make('state_id')
                    ->label(__('order.state'))
                    ->relationship('state', 'name')
                    ->searchable()
                    ->preload(),

                SelectFilter::make('created_by')
                    ->label(__('common.created_by'))
                    ->relationship('createdBy', 'name')
                    ->searchable()
                    ->preload(),

                Filter::make('start_date')
                    ->label(__('order.start_date_range'))
                    ->schema([
                        DatePicker::make('start_date_from')
                            ->label(__('order.from')),
                        DatePicker::make('start_date_until')
                            ->label(__('order.until')),
                    ])
                    ->query(function ($query, array $data) {
                        return $query
                            ->when(
                                $data['start_date_from'],
                                fn ($query, $date) => $query->whereDate('start_date', '>=', $date),
                            )
                            ->when(
                                $data['start_date_until'],
                                fn ($query, $date) => $query->whereDate('start_date', '<=', $date),
                            );
                    }),

                TrashedFilter::make(),
            ])
            ->recordActions([
                ViewAction::make(),
                EditAction::make(),
                DeleteAction::make(),
                ForceDeleteAction::make(),
                RestoreAction::make(),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ])
            ->defaultSort('created_at', 'desc')
            ->poll('30s');
    }
}
